/**
  ******************************************************************************
  * @file    py32f410_hal_rcc_ex.h
  * @author  MCU Application Team
  * @brief   Header file of RCC HAL Extended module.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __PY32F410_HAL_RCC_EX_H
#define __PY32F410_HAL_RCC_EX_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "py32f410_hal_def.h"

/** @addtogroup PY32F410_HAL_Driver
  * @{
  */

/** @addtogroup RCCEx
  * @{
  */

/* Private Constants ---------------------------------------------------------*/
/** @addtogroup RCCEx_Private_Constants
 * @{
 */

/**
  * @}
  */

/* Private Macros ------------------------------------------------------------*/
/** @addtogroup RCCEx_Private_Macros
 * @{
 */

#define IS_RCC_PLL_MUL(__MUL__) (((__MUL__) >= 2U) && ((__MUL__) <= 9U))

#define IS_RCC_MCOSOURCE(__SOURCE__) (((__SOURCE__) == RCC_MCOSOURCE_NOCLOCK)  || ((__SOURCE__) == RCC_MCOSOURCE_LSE)    || \
                                       ((__SOURCE__) == RCC_MCOSOURCE_LSI)      || \
                                       ((__SOURCE__) == RCC_MCOSOURCE_SYSCLK)   || ((__SOURCE__) == RCC_MCOSOURCE_HSI) || \
                                       ((__SOURCE__) == RCC_MCOSOURCE_HSE)      || ((__SOURCE__) == RCC_MCOSOURCE_PLLCLK) || \
                                       ((__SOURCE__) == RCC_MCOSOURCE_HCLK)     || ((__SOURCE__) == RCC_MCOSOURCE_PCLK1)  || \
                                       ((__SOURCE__) == RCC_MCOSOURCE_PCLK2)    || ((__SOURCE__) == RCC_MCOSOURCE_HSI10M))

#define IS_RCC_ADCCLKSOURCE(__ADCCLK__) (((__ADCCLK__) == RCC_ADCCLKSOURCE_SYSCLK)  || ((__ADCCLK__) == RCC_ADCCLKSOURCE_PLL)   || \
                                          ((__ADCCLK__) == RCC_ADCCLKSOURCE_HSI)     || ((__ADCCLK__) == RCC_ADCCLKSOURCE_HSE))

#define IS_RCC_PVDFILTERCLK(__PVDFILTERCLK__) (((__PVDFILTERCLK__) == RCC_PVDFILTERCLK_LSI)  || ((__PVDFILTERCLK__) == RCC_PVDFILTERCLK_LSE)  || \
                                               ((__PVDFILTERCLK__) == RCC_PVDFILTERCLK_PCLK1))

#define IS_RCC_USART1CLKSOURCE(__SOURCE__) (((__SOURCE__) == RCC_USART1CLKSOURCE_PCLK2)  || ((__SOURCE__) == RCC_USART1CLKSOURCE_SYSCLK) || \
                                            ((__SOURCE__) == RCC_USART1CLKSOURCE_LSE)    || ((__SOURCE__) == RCC_USART1CLKSOURCE_HSI))

#define IS_RCC_LPUART1CLKSOURCE(__SOURCE__) (((__SOURCE__) == RCC_LPUART1CLKSOURCE_PCLK1)  || ((__SOURCE__) == RCC_LPUART1CLKSOURCE_SYSCLK) || \
                                             ((__SOURCE__) == RCC_LPUART1CLKSOURCE_LSE)    || ((__SOURCE__) == RCC_LPUART1CLKSOURCE_HSI))

#define IS_RCC_LPTIM1CLKSOURCE(__SOURCE__) (((__SOURCE__) == RCC_LPTIM1CLKSOURCE_PCLK1) || ((__SOURCE__) == RCC_LPTIM1CLKSOURCE_LSI)   || \
                                            ((__SOURCE__) == RCC_LPTIM1CLKSOURCE_HSI)   || ((__SOURCE__) == RCC_LPTIM1CLKSOURCE_LSE))

#define IS_RCC_COMP1CLKSOURCE(__SOURCE__) (((__SOURCE__) == RCC_COMP1CLKSOURCE_PCLK2) || ((__SOURCE__) == RCC_COMP1CLKSOURCE_LSI)   || \
                                           ((__SOURCE__) == RCC_COMP1CLKSOURCE_LSE))

#define IS_RCC_COMP2CLKSOURCE(__SOURCE__) (((__SOURCE__) == RCC_COMP2CLKSOURCE_PCLK2) || ((__SOURCE__) == RCC_COMP2CLKSOURCE_LSI)   || \
                                           ((__SOURCE__) == RCC_COMP2CLKSOURCE_LSE))

#define IS_RCC_PERIPHCLOCK(PERIPHCLOCK)  ((PERIPHCLOCK) <= RCC_PERIPHCLK_MASK)

/**
  * @}
  */

/* Exported types ------------------------------------------------------------*/

/** @defgroup RCCEx_Exported_Types RCCEx Exported Types
  * @{
  */

/**
  * @brief  RCC extended clocks structure definition
  */
typedef struct
{
  uint32_t PeriphClockSelection;            /*!< The Extended Clock to be configured.
                                            This parameter can be a value of @ref RCCEx_Periph_Clock_Selection */

  uint32_t RtcClockSelection;              /*!< specifies the RTC clock source.
                                             This parameter can be a value of @ref RCC_RTC_Clock_Source */
  
  uint32_t AdcClockSelection;              /*!< ADC clock source
                                            This parameter can be a value of @ref RCCEx_ADC_Prescaler */
  
  uint32_t PvdFilterClockSelection;        /*!< PVD filter clock source
                                            This parameter can be a value of @ref RCCEx_PVD_Filter_Clock_Source */
  
  uint32_t Usart1ClockSelection;           /*!< Specifies USART1 clock source.
                                            This parameter can be a value of @ref RCCEx_USART1_Clock_Source */

  uint32_t Lpuart1ClockSelection;          /*!< Specifies LPUART1 clock source.
                                            This parameter can be a value of @ref RCCEx_LPUART1_Clock_Source */

  uint32_t Lptim1ClockSelection;           /*!< Specifies LPTIM1 clock source.
                                            This parameter can be a value of @ref RCCEx_LPTIM1_Clock_Source */
  
  uint32_t Comp1ClockSelection;            /*!< Specifies COMP1 clock source.
                                            This parameter can be a value of @ref RCCEx_COMP1_Clock_Source */
  
  uint32_t Comp2ClockSelection;            /*!< Specifies COMP2 clock source.
                                            This parameter can be a value of @ref RCCEx_COMP2_Clock_Source */

  uint32_t Opa2ClockSelection;            /*!< Specifies OPA2 clock source.
                                            This parameter can be a value of @ref RCCEx_OPA2_Clock_Source */
  
} RCC_PeriphCLKInitTypeDef;

/**
  * @}
  */

/* Exported constants --------------------------------------------------------*/
/** @defgroup RCCEx_Exported_Constants RCCEx Exported Constants
  * @{
  */

/** @defgroup RCCEx_Periph_Clock_Selection Periph Clock Selection
  * @{
  */
#define RCC_PERIPHCLK_RTC           0x00000001U
#define RCC_PERIPHCLK_ADC           0x00000002U
#define RCC_PERIPHCLK_PVD           0x00000004U
#define RCC_PERIPHCLK_USART1        0x00000008U
#define RCC_PERIPHCLK_LPUART1       0x00000010U
#define RCC_PERIPHCLK_LPTIM1        0x00000020U
#define RCC_PERIPHCLK_COMP1         0x00000040U
#define RCC_PERIPHCLK_COMP2         0x00000080U
#define RCC_PERIPHCLK_OPA2          0x00000100U

#define RCC_PERIPHCLK_MASK          0x000001FFU
/**
  * @}
  */

/** @defgroup RCCEx_ADC_Prescaler ADC Prescaler
  * @{
  */
#define RCC_ADCCLKSOURCE_SYSCLK              0x00000000
#define RCC_ADCCLKSOURCE_PLL                 RCC_CCIPR_ADC1SEL_0
#define RCC_ADCCLKSOURCE_HSI                 RCC_CCIPR_ADC1SEL_1
#define RCC_ADCCLKSOURCE_HSE                 RCC_CCIPR_ADC1SEL
/**
  * @}
  */

/** @defgroup RCCEx_PVD_Filter_Clock_Source PVD Filter Clock Source
  * @{
  */
#define RCC_PVDFILTERCLK_LSI            0x00000000
#define RCC_PVDFILTERCLK_LSE            RCC_CFGR1_PVDSEL_0
#define RCC_PVDFILTERCLK_PCLK1          RCC_CFGR1_PVDSEL_1
/**
  * @}
  */

/** @defgroup RCCEx_USART1_Clock_Source USART1 Clock Source
  * @{
  */
#define RCC_USART1CLKSOURCE_PCLK2      0x00000000U
#define RCC_USART1CLKSOURCE_SYSCLK     RCC_CCIPR_USART1SEL_0
#define RCC_USART1CLKSOURCE_HSI        RCC_CCIPR_USART1SEL_1
#define RCC_USART1CLKSOURCE_LSE        (RCC_CCIPR_USART1SEL_0 | RCC_CCIPR_USART1SEL_1)
/**
  * @}
  */

/** @defgroup RCCEx_LPUART1_Clock_Source LPUART1 Clock Source
  * @{
  */
#define RCC_LPUART1CLKSOURCE_PCLK1     0x00000000U
#define RCC_LPUART1CLKSOURCE_SYSCLK    RCC_CCIPR_LPUART1SEL_0
#define RCC_LPUART1CLKSOURCE_HSI       RCC_CCIPR_LPUART1SEL_1
#define RCC_LPUART1CLKSOURCE_LSE       (RCC_CCIPR_LPUART1SEL_0 | RCC_CCIPR_LPUART1SEL_1)
/**
  * @}
  */

/** @defgroup RCCEx_LPTIM1_Clock_Source LPTIM1 Clock Source
  * @{
  */
#define RCC_LPTIM1CLKSOURCE_PCLK1      0x00000000U
#define RCC_LPTIM1CLKSOURCE_LSI        RCC_CCIPR_LPTIM1SEL_0
#define RCC_LPTIM1CLKSOURCE_HSI        RCC_CCIPR_LPTIM1SEL_1
#define RCC_LPTIM1CLKSOURCE_LSE        RCC_CCIPR_LPTIM1SEL
/**
  * @}
  */

/** @defgroup RCCEx_COMP1_Clock_Source COMP1 Clock Source
  * @{
  */
#define RCC_COMP1CLKSOURCE_PCLK2      0x00000000U
#define RCC_COMP1CLKSOURCE_LSI        RCC_CCIPR_COMP1SEL_0
#define RCC_COMP1CLKSOURCE_LSE        RCC_CCIPR_COMP1SEL_1
/**
  * @}
  */

/** @defgroup RCCEx_COMP2_Clock_Source COMP2 Clock Source
  * @{
  */
#define RCC_COMP2CLKSOURCE_PCLK2      0x00000000U
#define RCC_COMP2CLKSOURCE_LSI        RCC_CCIPR_COMP2SEL_0
#define RCC_COMP2CLKSOURCE_LSE        RCC_CCIPR_COMP2SEL_1
/**
  * @}
  */

/** @defgroup RCCEx_OPA2_Clock_Source COMP2 Clock Source
  * @{
  */
#define RCC_OPA2CLKSOURCE_PCLK2      0x00000000U
#define RCC_OPA2CLKSOURCE_LSI        RCC_CCIPR_OPA2SEL_0
#define RCC_OPA2CLKSOURCE_LSE        RCC_CCIPR_OPA2SEL_1
/**
  * @}
  */



/** @defgroup RCC_MCO_Index MCO Index
  * @{
  */
#define RCC_MCO1                       0x00000000U
#define RCC_MCO                        RCC_MCO1               /*!< MCO1 to be compliant with other families with 2 MCOs*/
#define RCC_MCO2                       0x00000001U

/**
  * @}
  */

/** @defgroup RCCEx_MCO1_Clock_Source MCO1 Clock Source
  * @{
  */
#define RCC_MCOSOURCE_NOCLOCK           ((uint32_t)RCC_CFGR_MCO_NOCLOCK)
#define RCC_MCOSOURCE_LSE               ((uint32_t)RCC_CFGR_MCO_LSE)
#define RCC_MCOSOURCE_LSI               ((uint32_t)RCC_CFGR_MCO_LSI)
#define RCC_MCOSOURCE_SYSCLK            ((uint32_t)RCC_CFGR_MCO_SYSCLK)
#define RCC_MCOSOURCE_HSI               ((uint32_t)RCC_CFGR_MCO_HSI)
#define RCC_MCOSOURCE_HSE               ((uint32_t)RCC_CFGR_MCO_HSE)
#define RCC_MCOSOURCE_PLLCLK            ((uint32_t)RCC_CFGR_MCO_PLL)
#define RCC_MCOSOURCE_HCLK              ((uint32_t)RCC_CFGR_MCO_HCLK)
#define RCC_MCOSOURCE_PCLK1             ((uint32_t)RCC_CFGR_MCO_PCLK1)
#define RCC_MCOSOURCE_PCLK2             ((uint32_t)RCC_CFGR_MCO_PCLK2)
#define RCC_MCOSOURCE_HSI10M            ((uint32_t)RCC_CFGR_MCO_HSI10M)
/**
  * @}
  */

/**
  * @}
  */

/** @defgroup RCCEx_HSE_Configuration HSE Configuration
  * @{
  */

/**
  * @}
  */

/** @defgroup RCCEx_Peripheral_Configuration Peripheral Configuration
  * @brief  Macros to configure clock source of different peripherals.
  * @{
  */

/** @brief  Macro to configure the ADC clock.
  * @param  __ADCCLKSOURCE__ specifies the ADC clock source.
  *          This parameter can be one of the following values:
  *            @arg @ref RCC_ADCCLKSOURCE_SYSCLK  SYSCLK used as PVD filter clock
  *            @arg @ref RCC_ADCCLKSOURCE_PLL     PLL used as PVD filter clock
  *            @arg @ref RCC_ADCCLKSOURCE_HSI     HSI used as PVD filter clock
  *            @arg @ref RCC_ADCCLKSOURCE_HSE     HSE used as PVD filter clock
  */
#define __HAL_RCC_ADC_CONFIG(__ADCCLKSOURCE__) \
                  MODIFY_REG(RCC->CCIPR, RCC_CCIPR_ADC1SEL, (uint32_t)(__ADCCLKSOURCE__))

/** @brief  Macro to get the ADC clock.
  * @retval The clock source can be one of the following values:
  *            @arg @ref RCC_ADCCLKSOURCE_SYSCLK  SYSCLK used as PVD filter clock
  *            @arg @ref RCC_ADCCLKSOURCE_PLL     PLL used as PVD filter clock
  *            @arg @ref RCC_ADCCLKSOURCE_HSI     HSI used as PVD filter clock
  *            @arg @ref RCC_ADCCLKSOURCE_HSE     HSE used as PVD filter clock
  */
#define __HAL_RCC_GET_ADC_SOURCE() ((uint32_t)(READ_BIT(RCC->CCIPR, RCC_CCIPR_ADC1SEL)))


/** @brief  Macro to configure the PVD filter clock.
  * @param  __PVDFILTERCLK__ specifies the SRAM2 clock source.
  *          This parameter can be one of the following values:
  *            @arg @ref RCC_PVDFILTERCLK_LSI  LSI oscillator clock used as PVD filter clock
  *            @arg @ref RCC_PVDFILTERCLK_LSE  LSE oscillator clock used as PVD filter clock
  *            @arg @ref RCC_PVDFILTERCLK_PCLK1 PCLK  used as PVD filter clock
  */
#define __HAL_RCC_PVD_CONFIG(__PVDFILTERCLK__) \
                  MODIFY_REG(RCC->CFGR1, RCC_CFGR1_PVDSEL, (uint32_t)(__PVDFILTERCLK__))

/** @brief  Macro to get the PVD filter clock.
  * @retval The clock source can be one of the following values:
  *            @arg @ref RCC_PVDFILTERCLK_LSI  LSI oscillator clock used as PVD filter clock
  *            @arg @ref RCC_PVDFILTERCLK_LSE  LSE oscillator clock used as PVD filter clock
  *            @arg @ref RCC_PVDFILTERCLK_PCLK1 PCLK  used as PVD filter clock
  */
#define __HAL_RCC_GET_PVD_FILTER_SOURCE() ((uint32_t)(READ_BIT(RCC->CFGR1, RCC_CFGR1_PVDSEL)))

/** @brief  Macro to configure the USART1 clock (USART1CLK).
  *
  * @param  __USART1_CLKSOURCE__ specifies the USART1 clock source.
  *          This parameter can be one of the following values:
  *            @arg @ref RCC_USART1CLKSOURCE_PCLK2  PCLK2 selected as USART1 clock
  *            @arg @ref RCC_USART1CLKSOURCE_HSI  HSI selected as USART1 clock
  *            @arg @ref RCC_USART1CLKSOURCE_SYSCLK  System Clock selected as USART1 clock
  *            @arg @ref RCC_USART1CLKSOURCE_LSE  LSE selected as USART1 clock
  * @retval None
  */
#define __HAL_RCC_USART1_CONFIG(__USART1_CLKSOURCE__) \
                  MODIFY_REG(RCC->CCIPR, RCC_CCIPR_USART1SEL, (__USART1_CLKSOURCE__))

/** @brief  Macro to get the USART1 clock source.
  * @retval The clock source can be one of the following values:
  *            @arg @ref RCC_USART1CLKSOURCE_PCLK2  PCLK2 selected as USART1 clock
  *            @arg @ref RCC_USART1CLKSOURCE_HSI  HSI selected as USART1 clock
  *            @arg @ref RCC_USART1CLKSOURCE_SYSCLK  System Clock selected as USART1 clock
  *            @arg @ref RCC_USART1CLKSOURCE_LSE  LSE selected as USART1 clock
  */
#define __HAL_RCC_GET_USART1_SOURCE() (READ_BIT(RCC->CCIPR, RCC_CCIPR_USART1SEL))

/** @brief  Macro to configure the LPUART1 clock (LPUART1CLK).
  *
  * @param  __LPUART1_CLKSOURCE__ specifies the LPUART1 clock source.
  *          This parameter can be one of the following values:
  *            @arg @ref RCC_LPUART1CLKSOURCE_PCLK1  PCLK1 selected as LPUART1 clock
  *            @arg @ref RCC_LPUART1CLKSOURCE_HSI  HSI selected as LPUART1 clock
  *            @arg @ref RCC_LPUART1CLKSOURCE_SYSCLK  System Clock selected as LPUART1 clock
  *            @arg @ref RCC_LPUART1CLKSOURCE_LSE  LSE selected as LPUART1 clock
  * @retval None
  */
#define __HAL_RCC_LPUART1_CONFIG(__LPUART1_CLKSOURCE__) \
                  MODIFY_REG(RCC->CCIPR, RCC_CCIPR_LPUART1SEL, (__LPUART1_CLKSOURCE__))

/** @brief  Macro to get the LPUART1 clock source.
  * @retval The clock source can be one of the following values:
  *            @arg @ref RCC_LPUART1CLKSOURCE_PCLK1  PCLK1 selected as LPUART1 clock
  *            @arg @ref RCC_LPUART1CLKSOURCE_HSI  HSI selected as LPUART1 clock
  *            @arg @ref RCC_LPUART1CLKSOURCE_SYSCLK  System Clock selected as LPUART1 clock
  *            @arg @ref RCC_LPUART1CLKSOURCE_LSE  LSE selected as LPUART1 clock
  */
#define __HAL_RCC_GET_LPUART1_SOURCE() (READ_BIT(RCC->CCIPR, RCC_CCIPR_LPUART1SEL))

/** @brief  Macro to configure the LPTIM1 clock (LPTIM1CLK).
  *
  * @param  __LPTIM1_CLKSOURCE__ specifies the LPTIM1 clock source.
  *          This parameter can be one of the following values:
  *            @arg @ref RCC_LPTIM1CLKSOURCE_PCLK1  PCLK1 selected as LPTIM1 clock
  *            @arg @ref RCC_LPTIM1CLKSOURCE_LSI  LSI selected as LPTIM1 clock
  *            @arg @ref RCC_LPTIM1CLKSOURCE_HSI  HSI selected as LPTIM1 clock
  *            @arg @ref RCC_LPTIM1CLKSOURCE_LSE  LSE selected as LPTIM1 clock
  * @retval None
  */
#define __HAL_RCC_LPTIM1_CONFIG(__LPTIM1_CLKSOURCE__) \
                  MODIFY_REG(RCC->CCIPR, RCC_CCIPR_LPTIM1SEL, (__LPTIM1_CLKSOURCE__))

/** @brief  Macro to get the LPTIM1 clock source.
  * @retval The clock source can be one of the following values:
  *            @arg @ref RCC_LPTIM1CLKSOURCE_PCLK1  PCLK1 selected as LPTIM1 clock
  *            @arg @ref RCC_LPTIM1CLKSOURCE_LSI  LSI selected as LPTIM1 clock
  *            @arg @ref RCC_LPTIM1CLKSOURCE_HSI  HSI selected as LPTIM1 clock
  *            @arg @ref RCC_LPTIM1CLKSOURCE_LSE  LSE selected as LPTIM1 clock
  */
#define __HAL_RCC_GET_LPTIM1_SOURCE() (READ_BIT(RCC->CCIPR, RCC_CCIPR_LPTIM1SEL))

/** @brief  Macro to configure the COMP1 clock (COMP1CLK).
  *
  * @param  __COMP1_CLKSOURCE__ specifies the COMP1 clock source.
  *          This parameter can be one of the following values:
  *            @arg @ref RCC_COMP1CLKSOURCE_PCLK2  PCLK1 selected as COMP1 clock
  *            @arg @ref RCC_COMP1CLKSOURCE_LSI    HSI selected as COMP1 clock
  *            @arg @ref RCC_COMP1CLKSOURCE_LSE    LSE selected as COMP1 clock
  * @retval None
  */
#define __HAL_RCC_COMP1_CONFIG(__COMP1_CLKSOURCE__) \
                  MODIFY_REG(RCC->CCIPR, RCC_CCIPR_COMP1SEL, (__COMP1_CLKSOURCE__))

/** @brief  Macro to get the COMP1 clock source.
  * @retval The clock source can be one of the following values:
  *            @arg @ref RCC_COMP1CLKSOURCE_PCLK2  PCLK1 selected as COMP1 clock
  *            @arg @ref RCC_COMP1CLKSOURCE_LSI    HSI selected as COMP1 clock
  *            @arg @ref RCC_COMP1CLKSOURCE_LSE    LSE selected as COMP1 clock
  */
#define __HAL_RCC_GET_COMP1_SOURCE() (READ_BIT(RCC->CCIPR, RCC_CCIPR_COMP1SEL))

/** @brief  Macro to configure the COMP2 clock (COMP2CLK).
  *
  * @param  __COMP2_CLKSOURCE__ specifies the COMP2 clock source.
  *          This parameter can be one of the following values:
  *            @arg @ref RCC_COMP2CLKSOURCE_PCLK2  PCLK1 selected as COMP2 clock
  *            @arg @ref RCC_COMP2CLKSOURCE_LSI    HSI selected as COMP2 clock
  *            @arg @ref RCC_COMP2CLKSOURCE_LSE    LSE selected as COMP2 clock
  * @retval None
  */
#define __HAL_RCC_COMP2_CONFIG(__COMP2_CLKSOURCE__) \
                  MODIFY_REG(RCC->CCIPR, RCC_CCIPR_COMP2SEL, (__COMP2_CLKSOURCE__))

/** @brief  Macro to get the COMP2 clock source.
  * @retval The clock source can be one of the following values:
  *            @arg @ref RCC_COMP2CLKSOURCE_PCLK2  PCLK1 selected as COMP2 clock
  *            @arg @ref RCC_COMP2CLKSOURCE_LSI    HSI selected as COMP2 clock
  *            @arg @ref RCC_COMP2CLKSOURCE_LSE    LSE selected as COMP2 clock
  */
#define __HAL_RCC_GET_COMP2_SOURCE() (READ_BIT(RCC->CCIPR, RCC_CCIPR_COMP2SEL))

/** @brief  Macro to configure the OPA2 clock (OPA2CLK).
  *
  * @param  __OPA2_CLKSOURCE__ specifies the OPA2 clock source.
  *          This parameter can be one of the following values:
  *            @arg @ref RCC_OPA2CLKSOURCE_PCLK2  PCLK2 selected as OPA2 clock
  *            @arg @ref RCC_OPA2CLKSOURCE_LSI    LSI selected as OPA2 clock
  *            @arg @ref RCC_OPA2CLKSOURCE_LSE    LSE selected as OPA2 clock
  * @retval None
  */
#define __HAL_RCC_OPA2_CONFIG(__OPA2_CLKSOURCE__) \
                  MODIFY_REG(RCC->CCIPR, RCC_CCIPR_OPA2SEL, (__OPA2_CLKSOURCE__))

/** @brief  Macro to get the OPA2 clock source.
  * @retval The clock source can be one of the following values:
  *            @arg @ref RCC_OPA2CLKSOURCE_PCLK2  PCLK2 selected as OPA2 clock
  *            @arg @ref RCC_OPA2CLKSOURCE_LSI    LSI selected as OPA2 clock
  *            @arg @ref RCC_OPA2CLKSOURCE_LSE    LSE selected as OPA2 clock
  */
#define __HAL_RCC_GET_OPA2_SOURCE() (READ_BIT(RCC->CCIPR, RCC_CCIPR_OPA2SEL))


/**
  * @}
  */

/**
  * @}
  */

/* Exported functions --------------------------------------------------------*/
/** @addtogroup RCCEx_Exported_Functions
  * @{
  */

/** @addtogroup RCCEx_Exported_Functions_Group1
  * @{
  */

HAL_StatusTypeDef HAL_RCCEx_PeriphCLKConfig(RCC_PeriphCLKInitTypeDef  *PeriphClkInit);
void              HAL_RCCEx_GetPeriphCLKConfig(RCC_PeriphCLKInitTypeDef  *PeriphClkInit);
uint32_t          HAL_RCCEx_GetPeriphCLKFreq(uint32_t PeriphClk);

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* __PY32F410_HAL_RCC_EX_H */


/************************ (C) COPYRIGHT Puya *****END OF FILE******************/

